
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_SEP_ATP_ECC_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

CRYSError_t      TST_ATP_Error;
//static CRYS_ECPKI_KG_TempData_t    	TST_TempData={0};
//static CRYS_ECPKI_BUILD_TempData_t 	TST_BuildTempData={0};
//static CRYS_ECPKI_UserPrivKey_t    	TST_UserPrivKey1;
//static CRYS_ECPKI_UserPrivKey_t    	TST_UserPrivKey2;
//static CRYS_ECPKI_UserPublKey_t    	TST_UserPublKey1;
//static CRYS_ECPKI_UserPublKey_t    	TST_UserPublKey2;
//static CRYS_ECDH_TempData_t        	TST_ECDH_TempData={0}; 
//static CRYS_ECPKI_UserPublKey_t        TST_UserPublKeyExported; 
//static CRYS_ECDSA_SignUserContext_t    TST_SignUserContext;
//static CRYS_ECDSA_VerifyUserContext_t  TST_VerifyUserContext;
//static CRYS_EC_ELGAMAL_TempData_t  	TST_ELGAMAL_TempData;

/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   DxUint8_t   *FailureFunctionNameOut_ptr);

/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_NonIntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equally to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the signature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes Acceptance test for ECC ESDSA Sign & Verify
* Algorithm:  
*  1. Allocate memory buffers
*  2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
*  3. Call the CRYS_ECDSA_SignInit to init the Sign operation
*  4. Call the CRYS_ECDSA_SignUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
*  5. Call the CRYS_ECDSA_SignFinish to complete the sign operation
*  6. Call the CRYS_ECDSA_VerifyInit to init the verification operation
*  7. Call the CRYS_ECDSA_VerifyUpdate - This function is called N times
*                                        (N = number of blocks in data each block is 16byte)
*  8.Call the CRYS_ECDSA_VerifyFinish to complete the verification operation     
*  9. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ECDSA_NonIntegratedSignVerify(
	DxUint32_t			      KeyGenType,
	CRYS_ECPKI_DomainID_t     DomainID,
	DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
	DxUint32_t                PublKeySizeInBytes,  /*in*/
	DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
	DxUint32_t                PrivKeySizeInBytes,/*in*/
	CRYS_ECPKI_HASH_OpMode_t  HashMode,
	DxUint8_t				 *DataIn_ptr,
	DxUint32_t                DataInSize,
	DxUint32_t                SignatureSize,
	DxUint8_t                *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	DxUint32_t					NumOfBlocks = 0;
	DxUint32_t					Block_index = 0;
	CRYSError_t					TST_TestStatus=0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;
	CRYS_ECPKI_UserPrivKey_t	*TST_UserPrivKey;
	CRYS_ECPKI_UserPublKey_t	*TST_UserPublKey;
	CRYS_ECPKI_KG_TempData_t    *TST_TempData;
	CRYS_ECPKI_BUILD_TempData_t *TST_BuildTempData;
	CRYS_ECDSA_SignUserContext_t	*TST_SignUserContext;
	CRYS_ECDSA_VerifyUserContext_t  *TST_VerifyUserContext;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn			= DX_VOS_MemMalloc(DataInSize);
	pDataOut		= DX_VOS_MemMalloc(SignatureSize);
	TST_UserPrivKey = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_TempData    = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_KG_TempData_t));
	TST_BuildTempData	   = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_BUILD_TempData_t));
	TST_SignUserContext	   = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_SignUserContext_t));
	TST_VerifyUserContext  = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_VerifyUserContext_t));

	if ((TST_UserPrivKey == DX_NULL)     ||
		(TST_UserPublKey == DX_NULL)      ||
		(TST_TempData == DX_NULL)      ||
		(TST_BuildTempData == DX_NULL)      ||
		(TST_SignUserContext == DX_NULL)      ||
		(TST_VerifyUserContext == DX_NULL)      ||
		(pDataIn == DX_NULL)             ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/*Cleaning the temp data*/ 
	DX_VOS_MemSet(TST_TempData,0,sizeof(CRYS_ECPKI_KG_TempData_t));
	DX_VOS_MemSet(TST_BuildTempData,0,sizeof(CRYS_ECPKI_BUILD_TempData_t));

	/* 2. Prepare key */
	/***************/
	if (KeyGenType == TST_INTERNAL_KG) 
	{
		/* Generating the ECC key Pair */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(DomainID,   
			TST_UserPrivKey,
			TST_UserPublKey,   
			TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECPKI_GenKeyPair",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else
	{
		/* Building the ECC public Key */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
			DomainID,
			PublKeyIn_ptr,
			PublKeySizeInBytes,
			TST_UserPublKey,
			TST_BuildTempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECPKI_BuildPublKeyFullCheck",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key */
		/*------------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(
			DomainID,   
			PrivKeyIn_ptr,
			PrivKeySizeInBytes,
			TST_UserPrivKey);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECPKI_BuildPrivKey",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}

	/* 3. Initializing the Sign operation */
	/**************************************/
	TST_ATP_Error = CRYS_ECDSA_SignInit(
								TST_SignUserContext,
								TST_UserPrivKey,
								HashMode);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_ECDSA_SignInit",
		FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* Sign the data using the private Key */
	/* 4. Calling the update function         */
	/******************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS;

	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/* ECDSA SignUpdate block loop*/
	{ 
		TST_ATP_Error = CRYS_ECDSA_SignUpdate(TST_SignUserContext,
			pDataIn + (TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
			TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECDSA_SignUpdate",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}                       

	/* 5. Finish the Sign operation */
	/********************************/
	TST_ATP_Error = CRYS_ECDSA_SignFinish(TST_SignUserContext,
		pDataOut,
		&SignatureSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_ECDSA_SignFinish",
		FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 6. Verify the data using the public Key */
	/* Initializing the Verify operation       */
	/*******************************************/
	TST_ATP_Error = CRYS_ECDSA_VerifyInit(TST_VerifyUserContext,
		TST_UserPublKey,
		HashMode);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_ECDSA_VerifyInit",
		FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 7. Calling the update function */
	/**********************************/
	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/* ECDSA VerifyUpdate block loop*/
	{ 
		TST_ATP_Error = CRYS_ECDSA_VerifyUpdate(TST_VerifyUserContext,
			pDataIn+(TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
			TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECDSA_VerifyUpdate",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	} 

	/* 8. Finish the Verify operation */
	/**********************************/
	TST_ATP_Error = CRYS_ECDSA_VerifyFinish(
								TST_VerifyUserContext,
								pDataOut,
								SignatureSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_ECDSA_VerifyFinish",
		FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*9. Free allocated buffers */
	/****************************/
End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	DX_VOS_MemFree(TST_UserPrivKey);
	DX_VOS_MemFree(TST_UserPublKey);
	DX_VOS_MemFree(TST_TempData);
	DX_VOS_MemFree(TST_BuildTempData);
	DX_VOS_MemFree(TST_SignUserContext);
	DX_VOS_MemFree(TST_VerifyUserContext);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_IntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes Acceptance test for ECC ESDSA Sign & Verify
* Algorithm:  
*  1. Allocate memory buffers
*  2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
*  3. Call CRYS_ECDSA_Sign to perform the sign operation
*  4. Call CRYS_ECDSA_Verify to perform the verification operation   
*  5. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ECDSA_IntegratedSignVerify(
					DxUint32_t			      KeyGenType,
					CRYS_ECPKI_DomainID_t     DomainID,
					DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
					DxUint32_t                PublKeySizeInBytes,  /*in*/
					DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
					DxUint32_t                PrivKeySizeInBytes,/*in*/
					CRYS_ECPKI_HASH_OpMode_t  HashMode,
					DxUint8_t				 *DataIn_ptr,
					DxUint16_t                DataInSize,
					DxUint32_t                SignatureSize,
					DxUint8_t                *FailureFunctionNameOut_ptr)
{
	CRYSError_t						TST_TestStatus=0;
	DxUint8_t						*pDataIn;
	DxUint8_t						*pDataOut;
	CRYS_ECPKI_UserPrivKey_t		*TST_UserPrivKey;
	CRYS_ECPKI_UserPublKey_t		*TST_UserPublKey;
	CRYS_ECPKI_KG_TempData_t		*TST_TempData;
	CRYS_ECPKI_BUILD_TempData_t		*TST_BuildTempData;
	CRYS_ECDSA_SignUserContext_t	*TST_SignUserContext;
	CRYS_ECDSA_VerifyUserContext_t  *TST_VerifyUserContext;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn                = DX_VOS_MemMalloc(DataInSize);
	pDataOut               = DX_VOS_MemMalloc(SignatureSize);
	TST_UserPrivKey        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_TempData		   = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_KG_TempData_t));
	TST_BuildTempData	   = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_BUILD_TempData_t));
	TST_SignUserContext	   = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_SignUserContext_t));
	TST_VerifyUserContext  = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_VerifyUserContext_t));

	if ((TST_UserPrivKey == DX_NULL)      ||
		(TST_UserPublKey == DX_NULL)      ||
		(TST_TempData == DX_NULL)         ||
		(TST_BuildTempData == DX_NULL)    ||
		(TST_SignUserContext == DX_NULL)  ||
		(TST_VerifyUserContext == DX_NULL)||
		(pDataIn == DX_NULL)              ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}


	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/*Cleaning the temp data*/ 
	DX_VOS_MemSet(TST_TempData,0,sizeof(CRYS_ECPKI_KG_TempData_t));
	DX_VOS_MemSet(TST_BuildTempData,0,sizeof(CRYS_ECPKI_BUILD_TempData_t));

	/* 2. Prepare key */
	/******************/
	if (KeyGenType == TST_INTERNAL_KG) 
	{
		/* Generating the ECC key Pair */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(
						DomainID,   
						TST_UserPrivKey,
						TST_UserPublKey,   
						TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_GenKeyPair",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else
	{
		/* Building the ECC public Key */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
												DomainID,			
												PublKeyIn_ptr,
												PublKeySizeInBytes,
												TST_UserPublKey,
												TST_BuildTempData); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPublKeyFullCheck",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key */
		/*------------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(
									DomainID,   
									PrivKeyIn_ptr,
									PrivKeySizeInBytes,
									TST_UserPrivKey);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPrivKey",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

	}

	/* 3. Performing the Sign operation */
	/************************************/
	TST_ATP_Error = CRYS_ECDSA_Sign(TST_SignUserContext,
									TST_UserPrivKey,
									HashMode,                
									pDataIn,      
									DataInSize,
									pDataOut,
									&SignatureSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDSA_Sign",
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 4. Verify the data using the public Key */
	/*******************************************/
	TST_ATP_Error = CRYS_ECDSA_Verify(TST_VerifyUserContext,
									  TST_UserPublKey,
									  HashMode,
									  pDataOut,
									  SignatureSize,
									  pDataIn, 
									  DataInSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDSA_Verify",
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*9. Free allocated buffers */
	/****************************/
End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	DX_VOS_MemFree(TST_UserPrivKey);
	DX_VOS_MemFree(TST_UserPublKey);
	DX_VOS_MemFree(TST_TempData);
	DX_VOS_MemFree(TST_BuildTempData);
	DX_VOS_MemFree(TST_SignUserContext);
	DX_VOS_MemFree(TST_VerifyUserContext);

	return TST_TestStatus;	
}

	
/*********************************************************
* Function Name:                                                           
*  SEPQA_ECC_DH        
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn1_ptr   - The pointer to external public key data for user1.
* PublKeySizeInBytes1 -	Size of public key data in bytes 
* PublKeyIn2_ptr   - The pointer to external public key data for user2.
* PublKeySizeInBytes2 -	Size of public key data in bytes 
* PrivKeyIn1_ptr -      Pointer to private key data for user1. 
* PrivKeySizeInBytes1 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* PrivKeyIn2_ptr -      Pointer to private key data for user2. 
* PrivKeySizeInBytes2 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* MaxSecretKeySize    = maximal size of the shared secret key
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                      
* This function is qa external application on the SEP, 
* that executes Diffie-Hellman key shared secret generation using ECC capabilities
* Algorithm: 
* 1. Allocate memory buffers 
* 2.Build or generate the private and public keys for user 1 using       
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
* 3.Build or generate the private and public keys for user 2 using 
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
* 4. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 1 for user 1 
*    using user1 private key and user 2 public key 
* 5. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 2 for user 2 
*    using user2 private key and user 1 public key
* 6. Compare the two secret keys - to verify they are identical
* 7. Free allocated buffers
*
******************************************************************************/                                          
CRYSError_t SEPQA_ECC_DH(
				DxUint32_t			      KeyGenType,
				CRYS_ECPKI_DomainID_t     DomainID,
				DxUint8_t			     *PublKeyIn1_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes1,  /*in*/
				DxUint8_t			     *PublKeyIn2_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes2,  /*in*/
				DxUint8_t			     *PrivKeyIn1_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes1,/*in*/
				DxUint8_t			     *PrivKeyIn2_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes2,/*in*/
				DxUint32_t                MaxSecretKeySize,
				DxUint8_t                *FailureFunctionNameOut_ptr)
{	
	CRYSError_t           TST_TestStatus=0;
	DxUint8_t	         *pShareSecret1;
	DxUint8_t	         *pShareSecret2;
	DxUint32_t            TST_ShareSecret1Size = MaxSecretKeySize;
	DxUint32_t            TST_ShareSecret2Size = MaxSecretKeySize;

	CRYS_ECPKI_UserPrivKey_t		*TST_UserPrivKey1;
	CRYS_ECPKI_UserPublKey_t		*TST_UserPublKey1;
	CRYS_ECPKI_UserPrivKey_t		*TST_UserPrivKey2;
	CRYS_ECPKI_UserPublKey_t		*TST_UserPublKey2;
	CRYS_ECPKI_KG_TempData_t		*TST_TempData;
	CRYS_ECPKI_BUILD_TempData_t		*TST_BuildTempData;
	CRYS_ECDSA_SignUserContext_t	*TST_SignUserContext;
	CRYS_ECDSA_VerifyUserContext_t  *TST_VerifyUserContext;
	CRYS_ECDH_TempData_t        	*TST_ECDH_TempData;
	

	/* 1. Allocate memory buffers */
	/******************************/
	pShareSecret1 = DX_VOS_MemMalloc(MaxSecretKeySize);
	pShareSecret2 = DX_VOS_MemMalloc(MaxSecretKeySize);

	TST_UserPrivKey1        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey1        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_UserPrivKey2        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey2        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_TempData		   = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_KG_TempData_t));
	TST_ECDH_TempData	   = DX_VOS_MemMalloc(sizeof(CRYS_ECDH_TempData_t));
	TST_BuildTempData	   = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_BUILD_TempData_t));
	TST_SignUserContext	   = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_SignUserContext_t));
	TST_VerifyUserContext  = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_VerifyUserContext_t));
	

	if ((TST_UserPrivKey1 == DX_NULL)     ||
		(TST_UserPublKey1 == DX_NULL)     ||
		(TST_UserPrivKey2 == DX_NULL)     ||
		(TST_UserPublKey2 == DX_NULL)     ||
		(TST_TempData == DX_NULL)         ||
		(TST_ECDH_TempData == DX_NULL)    ||
		(TST_BuildTempData == DX_NULL)    ||
		(TST_SignUserContext == DX_NULL)  ||
		(TST_VerifyUserContext == DX_NULL)||
		(pShareSecret1 == DX_NULL)        ||
		(pShareSecret2 == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	/*Cleaning the temp data*/ 
	DX_VOS_MemSet(TST_TempData,0,sizeof(CRYS_ECPKI_KG_TempData_t));
	DX_VOS_MemSet(TST_BuildTempData,0,sizeof(CRYS_ECPKI_BUILD_TempData_t));
	DX_VOS_MemSet(TST_ECDH_TempData,0,sizeof(CRYS_ECDH_TempData_t));
	
	/* 2.Build or generate the private and public keys for user 1 */
	/**************************************************************/

	/* 3.Build or generate the private and public keys for user 2 */
	/**************************************************************/
	if (KeyGenType == TST_INTERNAL_KG) 
	{
		/* Generating the ECC key Pair for user 1 */
		/******************************************/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(
											DomainID,   
											TST_UserPrivKey1,
											TST_UserPublKey1,   
											TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_GenKeyPair",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Generating the ECC key Pair for user 2 */
		/******************************************/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(
										DomainID,   
										TST_UserPrivKey2,
										TST_UserPublKey2,   
										TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_GenKeyPair",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else
	{
		/* Building the ECC public Key for user 1 */
		/******************************************/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
													DomainID,			
													PublKeyIn1_ptr,
													PublKeySizeInBytes1,
													TST_UserPublKey1,
													TST_BuildTempData ); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPublKeyFullCheck",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC public Key for user 2 */
		/******************************************/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
													DomainID,			
													PublKeyIn2_ptr,
													PublKeySizeInBytes2,
													TST_UserPublKey2,
													TST_BuildTempData ); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPublKeyFullCheck",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key for user 1 */
		/*-----------------------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(DomainID,   
												PrivKeyIn1_ptr,
												PrivKeySizeInBytes1,
												TST_UserPrivKey1);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPrivKey",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key for user 2 */
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(DomainID,   
												PrivKeyIn2_ptr,
												PrivKeySizeInBytes2,
												TST_UserPrivKey2);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPrivKey",
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}

	/* 4. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 1 for user 1 */
	/*********************************************************************/
	TST_ATP_Error = CRYS_ECDH_SVDP_DH(TST_UserPublKey2,
									  TST_UserPrivKey1,
									  pShareSecret1,
									  &TST_ShareSecret1Size,
									  TST_ECDH_TempData);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDH_SVDP_DH",
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 2 for user 2 */
	/*********************************************************************/
	TST_ATP_Error = CRYS_ECDH_SVDP_DH(TST_UserPublKey1,
									  TST_UserPrivKey2,
									  pShareSecret2,
									  &TST_ShareSecret2Size,
									  TST_ECDH_TempData);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDH_SVDP_DH",
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 6. Verifying we have the same Secret Key both for client & Server */
	/******************************************************************/                                                    
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pShareSecret1 ,
											   pShareSecret2,
											   TST_ShareSecret2Size);   

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp - shared secret", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*7. Free allocated buffers */
	/****************************/
End:
	DX_VOS_MemFree(pShareSecret1);
	DX_VOS_MemFree(pShareSecret2);

	DX_VOS_MemFree(TST_UserPrivKey1);
	DX_VOS_MemFree(TST_UserPublKey1);
	DX_VOS_MemFree(TST_UserPrivKey2);
	DX_VOS_MemFree(TST_UserPublKey2);
	DX_VOS_MemFree(TST_TempData);
	DX_VOS_MemFree(TST_ECDH_TempData);
	DX_VOS_MemFree(TST_BuildTempData);
	DX_VOS_MemFree(TST_SignUserContext);
	DX_VOS_MemFree(TST_VerifyUserContext);

	return TST_TestStatus;	
}

/*********************************************************
* Function Name:                                                           
*  SEPQA_ECC_ElGamal_EncDec        
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data for user1.
* PublKeySizeInBytes -	Size of public key data in bytes 
* PrivKeyIn_ptr -      Pointer to private key data for user1. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr         - input data
* DataInSize         - data size
* EncryptedMsgSize   - encrypted message size
* MaxSecretKeySize    - maximal size of the shared secret key
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description: 
* This function is qa external application on the SEP, 
* that executes El-Gamal algorithm for encryption and decryption
*
* Algorithm:  
* 1. Allocate memory buffers 
* 2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
* 3. Encrypt a block of data using the CRYS_ECPKI_ELGAMAL_Encrypt encryption process
* 4. Decrypt a block of data using CRYS_ECPKI_ELGAMAL_Decrypt decryption process 
* 5. Compare the decrypted data to the original data
*
******************************************************************************/                                          
CRYSError_t SEPQA_ECC_ElGamal_EncDec(
						 DxUint32_t			      KeyGenType,
						 CRYS_ECPKI_DomainID_t    DomainID,
						 DxUint8_t			     *PublKeyIn_ptr,       
						 DxUint32_t               PublKeySizeInBytes,  
						 DxUint8_t			     *PrivKeyIn_ptr,     
						 DxUint32_t               PrivKeySizeInBytes,
					     DxUint8_t				 *DataIn_ptr,
						 DxUint16_t               DataInSize,
						 DxUint32_t               EncryptedMsgSize,
						 DxUint8_t               *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	CRYSError_t           TST_TestStatus=0;
	DxUint8_t	         *pDataIn;
	DxUint8_t            *pDataOut;            
	DxUint8_t            *pDecryptedData;
	DxUint32_t            DecryptedDataSize = DataInSize;
	CRYS_ECPKI_UserPrivKey_t		*TST_UserPrivKey1;
	CRYS_ECPKI_UserPublKey_t		*TST_UserPublKey1;
	CRYS_ECPKI_KG_TempData_t		*TST_TempData;
	CRYS_ECPKI_BUILD_TempData_t		*TST_BuildTempData;
	CRYS_EC_ELGAMAL_TempData_t  	*TST_ELGAMAL_TempData;
	
	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn			      = DX_VOS_MemMalloc(DataInSize);
	pDataOut		      = DX_VOS_MemMalloc(EncryptedMsgSize);
	pDecryptedData		  = DX_VOS_MemMalloc(DecryptedDataSize);
	TST_UserPrivKey1      = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey1	  = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_TempData          = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_KG_TempData_t));
	TST_BuildTempData     = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_BUILD_TempData_t));
	TST_ELGAMAL_TempData  = DX_VOS_MemMalloc(sizeof(CRYS_EC_ELGAMAL_TempData_t));

	
	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/*Cleaning the temp data*/ 
	DX_VOS_MemSet(TST_TempData,0,sizeof(CRYS_ECPKI_KG_TempData_t));
	DX_VOS_MemSet(TST_BuildTempData,0,sizeof(CRYS_ECPKI_BUILD_TempData_t));
	DX_VOS_MemSet(TST_ELGAMAL_TempData,0,sizeof(CRYS_EC_ELGAMAL_TempData_t));

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG) 
	{
		/* Generating the ECC key Pair */
		/*******************************/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(DomainID,   
											  TST_UserPrivKey1,
											  TST_UserPublKey1,   
											  TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_GenKeyPair", 
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else
	{
		/* Building the ECC public Key */
		/*******************************/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(DomainID,			
														 PublKeyIn_ptr,
														 PublKeySizeInBytes,
														 TST_UserPublKey1,
														 TST_BuildTempData ); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPublKeyFullCheck", 
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key */
		/***********************************/
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(DomainID,   
												PrivKeyIn_ptr,
												PrivKeySizeInBytes,
												TST_UserPrivKey1);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPrivKey", 
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}

	/* 3. Encrypting the message */
	/*****************************/
	TST_ATP_Error =  CRYS_ECPKI_ELGAMAL_Encrypt(TST_UserPublKey1,
												pDataIn,
												DataInSize,
												pDataOut,
												&EncryptedMsgSize,
												TST_ELGAMAL_TempData);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECPKI_ELGAMAL_Encrypt", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 4. Decrypting the massage */
	/*****************************/
	TST_ATP_Error =  CRYS_ECPKI_ELGAMAL_Decrypt(TST_UserPrivKey1,
												pDataOut,
												EncryptedMsgSize,
												pDecryptedData,
												&DecryptedDataSize,
												TST_ELGAMAL_TempData);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECPKI_ELGAMAL_Decrypt", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Comparing the decrypted data to the original */
	/***************************************************/                                                    
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDecryptedData,
											   pDataIn,
											   DataInSize);   

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp - decryption", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 6. Free allocated buffers */
	/*****************************/
End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(pDecryptedData);

	DX_VOS_MemFree(TST_UserPrivKey1);
	DX_VOS_MemFree(TST_UserPublKey1);
	DX_VOS_MemFree(TST_TempData);
	DX_VOS_MemFree(TST_BuildTempData);
	DX_VOS_MemFree(TST_ELGAMAL_TempData);

	return TST_TestStatus;	
}

/*********************************************************
* Function Name:                                                           
*  SEPQA_ECC_ImportExportKeyPair        
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data for user1.
* PublKeySizeInBytes -	Size of public key data in bytes 
* PrivKeyIn_ptr -      Pointer to private key data for user1. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* HashMode           - hash mode
* DataIn_ptr         - input data
* DataInSize         - data size
* SignatureSize      - size of signatured msg
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures

*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:     
* This function is qa external application on the SEP, 
* that executes ECC export public key functionality test
*
* Algorithm:  
*  1. Allocate memory buffers 
*  2. Prepare keys 
*     a. Generate keys 
*     or
*     a. Build the Private Key 
*     b. Build the Public Key 
*  3.	Export the public key using CRYS_ECPKI_ExportPublKey
*  4.	Re-build the exported key using CRYS_ECPKI_Build_PubKey
*  5.	Sign a massage using the exported public key using CRYS_ECDSA_Sign to perform the sign operation
*  6.	Verify the message using the appropriate public key using CRYS_ECDSA_Verify
*      to perform the verification operation
*  7. Free allocated buffers 
*
******************************************************************************/                                          
CRYSError_t SEPQA_ECC_ImportExportKeyPair(
									 DxUint32_t			      KeyGenType,
									 CRYS_ECPKI_DomainID_t    DomainID,
									 DxUint8_t			     *PublKeyIn_ptr,       
									 DxUint32_t               PublKeySizeInBytes,  
									 DxUint8_t			     *PrivKeyIn_ptr,     
									 DxUint32_t               PrivKeySizeInBytes,
									 CRYS_ECPKI_HASH_OpMode_t HashMode,
									 DxUint8_t				 *DataIn_ptr,
									 DxUint32_t               DataInSize,
									 DxUint32_t               SignatureSize,
									 DxUint8_t               *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	CRYSError_t						TST_TestStatus=0;
	DxUint8_t						*pDataIn;
	DxUint8_t						*pDataOut;
	DxUint8_t						TST_UserPublKeyExported_buff[TST_MAX_ECC_KEY_SIZE_BYTES]; 
	DxUint32_t						TST_UserPublKeyExported_buffSize;
	CRYS_ECPKI_UserPrivKey_t		*TST_UserPrivKey1;
	CRYS_ECPKI_UserPublKey_t		*TST_UserPublKey1;
	CRYS_ECPKI_KG_TempData_t		*TST_TempData;
	CRYS_ECPKI_BUILD_TempData_t     *TST_BuildTempData; 
	CRYS_ECPKI_UserPublKey_t        *TST_UserPublKeyExported;
	CRYS_ECDSA_SignUserContext_t    *TST_SignUserContext;
	CRYS_ECDSA_VerifyUserContext_t  *TST_VerifyUserContext;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn                 = DX_VOS_MemMalloc(DataInSize);
	pDataOut                = DX_VOS_MemMalloc(SignatureSize);
	TST_UserPrivKey1        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPrivKey_t));
	TST_UserPublKey1        = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_BuildTempData       = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_BUILD_TempData_t));
	TST_TempData            = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_KG_TempData_t));
	TST_UserPublKeyExported = DX_VOS_MemMalloc(sizeof(CRYS_ECPKI_UserPublKey_t));
	TST_SignUserContext     = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_SignUserContext_t));
	TST_VerifyUserContext   = DX_VOS_MemMalloc(sizeof(CRYS_ECDSA_VerifyUserContext_t));

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/*Cleaning the temp data*/ 
	DX_VOS_MemSet(TST_TempData,0,sizeof(CRYS_ECPKI_KG_TempData_t));
	DX_VOS_MemSet(TST_BuildTempData,0,sizeof(CRYS_ECPKI_BUILD_TempData_t));

	/*Initializing the data*/
	TST_UserPublKeyExported_buffSize = TST_MAX_ECC_KEY_SIZE_BYTES;

	/* 2. Prepare keys */
	/*******************/
	if (KeyGenType == TST_INTERNAL_KG) 
	{
		/* Generating the ECC key Pair */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_GenKeyPair(
								DomainID,   
								TST_UserPrivKey1,
								TST_UserPublKey1,   
								TST_TempData);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_GenKeyPair", 
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else
	{
		/* Building the ECC public Key */
		/*-----------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
										DomainID,			
										PublKeyIn_ptr,
										PublKeySizeInBytes,
										TST_UserPublKey1,
										TST_BuildTempData ); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_ECPKI_BuildPublKeyFullCheck", 
													FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

		/* Building the ECC private Key */
		/*------------------------------*/
		TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(
										DomainID,   
										PrivKeyIn_ptr,
										PrivKeySizeInBytes,
										TST_UserPrivKey1);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_ECPKI_BuildPrivKey", 
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	/* 3. Export the public key the massage */
	/****************************************/
	TST_ATP_Error =  CRYS_ECPKI_ExportPublKey(TST_UserPublKey1,
											  CRYS_EC_PointUncompressed,
											  TST_UserPublKeyExported_buff,
											  &TST_UserPublKeyExported_buffSize); 

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECPKI_ExportPublKey", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/*4. Build the exported key to a different pubkey*/
	/*************************************************/
	TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
											DomainID,			
											TST_UserPublKeyExported_buff,
											TST_UserPublKeyExported_buffSize,
											TST_UserPublKeyExported,
											TST_BuildTempData ); 

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECPKI_BuildPublKeyFullCheck", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Performing the Sign operation */
	/************************************/
	TST_ATP_Error = CRYS_ECDSA_Sign(TST_SignUserContext,
									TST_UserPrivKey1,
									HashMode,                
									pDataIn,      
									DataInSize,
									pDataOut,
									&SignatureSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDSA_Sign", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 6. Verify the data using the exported public Key */
	/****************************************************/
	TST_ATP_Error = CRYS_ECDSA_Verify(TST_VerifyUserContext,
									  TST_UserPublKeyExported,
									  HashMode,
									  pDataOut,
									  SignatureSize,
									  pDataIn, 
									  DataInSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_ECDSA_Verify with exported key", 
												FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 7. Free allocated buffers */
	/*****************************/
End:
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	DX_VOS_MemFree(TST_UserPrivKey1);
	DX_VOS_MemFree(TST_UserPublKey1);
	DX_VOS_MemFree(TST_BuildTempData);
	DX_VOS_MemFree(TST_TempData);
	DX_VOS_MemFree(TST_UserPublKeyExported);
	DX_VOS_MemFree(TST_SignUserContext);
	DX_VOS_MemFree(TST_VerifyUserContext);
	
	return TST_TestStatus;	

}









